# TeaHouse Hero Block (`emrikol/teahouse-hero`)

A hero/cover block whose background image updates on the client based on the visitor's local time, with flexible mapping and timezone offset options.
Backgrounds update every minute via preloaded images from configurable image directories, with robust error handling, fade transitions, and advanced visual enhancements.

## Features

### Core Time-Based Functionality
- **Dynamic backgrounds**: Image updates every minute according to mapped filenames with special case handling (03:14 and 03:16+) and even/odd hour logic
- **Timezone support**: Optional timezone offset adjustment for global deployment
- **Graceful fallbacks**: If an image is missing, previous background remains with warnings in console and editor
- **Live previews**: Editor and frontend use identical mapping/preload logic for accurate previews

### Visual Enhancement System
- **Upscaled image support**: Choose between standard `images/` or high-resolution `images-upscaled/` directories
- **Advanced noise overlay**: Procedural noise textures to enhance image appearance when enlarged
  - Fractal noise or turbulence patterns
  - Configurable opacity, scale, frequency, and octaves
  - Monochrome or full-color noise with custom color selection
  - Multiple blend modes (overlay, multiply, screen, etc.)
- **Overlay controls**: Color and opacity controls for text readability
- **Content alignment**: Top, center, or bottom content positioning

### Layout and Accessibility
- **Rich inner content**: Supports Heading, Paragraph, Buttons blocks by default with full InnerBlocks flexibility
- **Wide/full alignment**: Supports WordPress wide and full-width alignments
- **Accessibility**: Configurable `aria-label` for screen readers
- **Responsive design**: Optimized for all device sizes

## Usage

1. Add the "TeaHouse Hero" block anywhere in your content
2. Configure settings in the block sidebar:
   - **Background Images**: Toggle upscaled images, set timezone offset
   - **Content & Overlay**: Adjust content alignment, overlay color/opacity, noise effects
   - **Accessibility**: Set aria-label for screen readers
3. Add and arrange inner blocks (headings, text, buttons, etc.)

## Configuration Options

### Background Image Settings
- **Use Upscaled Images**: Toggle between `images/` and `images-upscaled/` directories
- **Timezone Offset (minutes)**: Adjust time calculation for different timezones

### Content & Overlay
- **Content Alignment**: Position content at top, center, or bottom
- **Overlay Color**: Background color for text readability
- **Overlay Opacity**: Transparency level (0-1)
- **Enable Noise Overlay**: Add procedural texture to enhance enlarged images

### Noise Overlay Options (when enabled)
- **Noise Opacity**: Intensity of noise effect (0-100%)
- **Noise Type**: Fractal Noise or Turbulence patterns
- **Noise Scale**: Size of noise particles (0-100%, lower = finer)
- **Noise Seed**: Random seed for different patterns (1-999)
- **Noise Color Type**: Monochrome or full-color noise
- **Noise Color**: Color for monochrome noise (when applicable)
- **Noise Blend Mode**: How noise blends with background (overlay, multiply, screen, etc.)
- **Noise Frequency**: Base frequency for pattern detail (0.1-2.0)
- **Noise Octaves**: Number of noise layers for complexity (1-8)

### Accessibility
- **Region Aria Label**: Custom accessibility label for screen readers

## Image Mapping Rules

The plugin uses JPG format with specific filename mapping:

- At **03:14** exactly: filename is `0314.jpg`
- At **03:16** or later until 03:59: `04.jpg`
- Even hours (0, 2, 4, 6, etc.): `<HH>.jpg` (zero-padded, e.g., `08.jpg`)
- Odd hours (1, 3, 5, 7, etc.): previous even hour `<HH-1>.jpg` (e.g., 15:30 → `14.jpg`)

If the mapped image doesn't exist, the previous background remains visible.

## Image Requirements

### Directory Structure
- **Standard images**: `images/` directory in your plugin folder
- **Upscaled images**: `images-upscaled/` directory for higher resolution versions
- Both directories should contain the same filenames as computed by mapping rules

### Required Images
Based on the mapping rules, you need these files:
- `0314.jpg` (for 3:14 AM exactly)
- `00.jpg`, `02.jpg`, `04.jpg`, `06.jpg`, `08.jpg`, `10.jpg`, `12.jpg`, `14.jpg`, `16.jpg`, `18.jpg`, `20.jpg`, `22.jpg`

### Image Guidelines
- Use consistent aspect ratios across all images
- Optimize for web delivery (reasonable file sizes)
- Consider providing both standard and upscaled versions for best quality

## Accessibility

- **Text readability**: Overlay color and opacity controls ensure readable text over any background
- **Semantic markup**: Block uses proper `region` role with optional `aria-label`
- **Screen reader support**: All controls properly labeled for assistive technologies
- **Keyboard navigation**: Full keyboard accessibility for all interactive elements

## FAQ

**Q: How often will my hero image update?**
A: Every minute, aligned to the local clock (or shifted by timezone offset if configured).

**Q: What happens if an image is missing?**
A: The previous image stays visible; a warning appears in the browser console and as an editor notice.

**Q: Should I use upscaled images?**
A: Yes, if you have higher resolution versions. The upscaled option is enabled by default and provides better quality for large displays.

**Q: What's the noise overlay for?**
A: It adds subtle texture to images that can improve visual quality when images are displayed larger than their original resolution.

**Q: Can I use different image formats?**
A: Currently, the plugin is optimized for JPG format. All images should use the `.jpg` extension.

**Q: How do I add my own images?**
A: Place your images in the plugin's `images/` directory (and optionally `images-upscaled/`) following the naming convention from the mapping rules.

## Development

### Prerequisites

- **Node.js**: Version 18+ recommended
- **PHP**: Version 7.4+ (as specified in composer.json)
- **Composer**: For PHP dependency management
- **WordPress**: Local development environment

### Setup

1. **Clone the repository**:
   ```bash
   git clone <repository-url>
   cd teahouse-hero
   ```

2. **Install dependencies**:
   ```bash
   # Install JavaScript dependencies
   npm install

   # Install PHP dependencies for code quality tools
   composer install
   ```

### Build Commands

The plugin uses WordPress Scripts for development workflow:

```bash
# Development build with file watching
npm run start

# Production build (optimized)
npm run build

# Lint JavaScript files
npm run lint:js

# Lint CSS/SCSS files
npm run lint:css

# Update WordPress packages
npm run packages-update

# Create plugin zip file for distribution
npm run plugin-zip
```

### Code Quality

#### JavaScript/CSS Linting
```bash
# Check JavaScript code style
npm run lint:js

# Check CSS/SCSS code style
npm run lint:css
```

#### PHP Code Standards
```bash
# Check PHP code standards (WordPress Coding Standards)
composer lint
# or
composer phpcs

# Auto-fix PHP code style issues
composer fix
# or
composer phpcbf

# Check specific files
composer phpcs path/to/file.php
composer phpcbf path/to/file.php
```

#### Testing
```bash
# Run PHP unit tests
composer test

# Run tests with coverage report
composer test:coverage
```

### Project Structure

```
teahouse-hero/
├── src/teahouse-hero/           # Source files
│   ├── block.json              # Block metadata and configuration
│   ├── index.js                # Block registration entry point
│   ├── edit.js                 # Editor component with controls
│   ├── save.js                 # Frontend save output
│   ├── view.js                 # Frontend JavaScript
│   ├── style.scss              # Frontend styles
│   ├── editor.scss             # Editor-specific styles
│   └── utils/                  # Utility modules
│       ├── time-utils.js       # Time mapping logic
│       ├── image-utils.js      # Image preloading utilities
│       └── noise-utils.js      # SVG noise generation
├── build/teahouse-hero/         # Compiled assets (auto-generated)
├── images/                     # Standard resolution images
├── images-upscaled/            # High resolution images
├── teahouse-hero.php           # Main plugin file
├── package.json                # Node.js dependencies and scripts
├── composer.json               # PHP dependencies and scripts
├── .phpcs.xml.dist             # PHP CodeSniffer configuration
└── README.md                   # This file
```

### Development Workflow

1. **Start development mode**:
   ```bash
   npm run start
   ```
   This watches for changes and rebuilds automatically.

2. **Make your changes** in the `src/` directory:
   - Edit JavaScript, CSS, or PHP files
   - Test in WordPress admin and frontend

3. **Lint and fix code**:
   ```bash
   # Fix JavaScript/CSS issues
   npm run lint:js
   npm run lint:css

   # Fix PHP issues
   composer fix
   composer lint
   ```

4. **Build for production**:
   ```bash
   npm run build
   ```

5. **Create distribution package**:
   ```bash
   npm run plugin-zip
   ```

### Key Files

- **`teahouse-hero.php`**: Main plugin file with block registration
- **`src/teahouse-hero/block.json`**: Block metadata, attributes, and WordPress integration
- **`src/teahouse-hero/edit.js`**: Editor interface with live preview and controls
- **`src/teahouse-hero/view.js`**: Frontend time-based background switching
- **`src/teahouse-hero/utils/time-utils.js`**: Core time mapping algorithm
- **`src/teahouse-hero/utils/noise-utils.js`**: SVG noise pattern generation

### Coding Standards

- **JavaScript**: Follows WordPress JavaScript coding standards via `@wordpress/scripts`
- **PHP**: Follows WordPress PHP coding standards (PHPCS with WPCS ruleset)
- **CSS**: Uses WordPress CSS guidelines
- **Accessibility**: WCAG 2.1 AA compliance required

### Contributing

1. Follow the existing code style and structure
2. Test changes in both editor and frontend environments
3. Ensure all linting passes before submitting
4. Update documentation for new features
5. Test with various timezone offsets and image configurations

### Debugging

- **Console logs**: Frontend JavaScript logs warnings for missing images
- **Editor notices**: Missing images show warnings in the block editor
- **PHP errors**: Check WordPress debug logs for server-side issues
- **Network tab**: Monitor image loading in browser developer tools

---

**Created by**: Derrick Tennant and Telex by WordPress.com
**License**: GPL-2.0-or-later
**Support**: See block editor sidebar for configuration help