import { __ } from '@wordpress/i18n';
import {
	useBlockProps,
	InnerBlocks,
	InspectorControls,
} from '@wordpress/block-editor';
import { useState, useEffect, useRef } from '@wordpress/element';
import './editor.scss';
import {
	PanelBody,
	TextControl,
	ToggleControl,
	RangeControl,
	SelectControl,
	ColorPalette,
	ToolbarGroup,
	ToolbarButton,
	Notice,
} from '@wordpress/components';
import { generateImageUrl } from './utils/time-utils';
import { preload, normalizeBasePath } from './utils/image-utils';
import { applyNoiseStyles } from './utils/noise-utils';
import { TIMERS, DEFAULTS } from './constants';

// Localized template and options
const LOCALIZED_TEMPLATE = [
	[
		'core/heading',
		{ placeholder: __( 'Hero heading…', 'teahouse-hero-block-wp' ) },
	],
	[
		'core/paragraph',
		{
			placeholder: __(
				'Hero description or message.',
				'teahouse-hero-block-wp'
			),
		},
	],
	[ 'core/buttons' ],
];

const LOCALIZED_ALIGN_OPTIONS = [
	{ label: __( 'Top', 'teahouse-hero-block-wp' ), value: 'top' },
	{ label: __( 'Center', 'teahouse-hero-block-wp' ), value: 'center' },
	{ label: __( 'Bottom', 'teahouse-hero-block-wp' ), value: 'bottom' },
];

// -------- Time mapping logic moved to utils/ ---------

export default function Edit( { attributes, setAttributes } ) {
	const {
		align,
		useUpscaled,
		timezoneOffsetMinutes,
		contentAlignment,
		overlayColor,
		overlayOpacity,
		ariaLabel,
		enableNoise,
		noiseOpacity,
		noiseScale,
		noiseType,
		noiseOctaves,
		noiseFrequency,
		noiseSeed,
		noiseColorType,
		noiseColor,
		noiseBlendMode,
	} = attributes;

	// Resolve to absolute plugin URL so editor doesn't resolve under /wp-admin/
	const normalizedBasePath = normalizeBasePath(
		useUpscaled ? 'images-upscaled/' : 'images/'
	);
	const normalizedExt = 'jpg'; // Always use JPG since we have JPG images
	const rootRef = useRef( null );

	// For background image management:
	const [ bgUrl, setBgUrl ] = useState( null );
	const [ isLoaded, setIsLoaded ] = useState( false );
	const [ warning, setWarning ] = useState( '' );
	const [ , setPreviewTime ] = useState( new Date() );

	// Timer management
	const timers = useRef( { timeout: null, interval: null } );

	// Compute filename for current and next minute, handle loading/preloading logic
	useEffect( () => {
		let cancelled = false;
		let lastGoodUrl = bgUrl; // In case of fallback on error

		function scheduleNextTick( now ) {
			const nextMinute = new Date(
				now.getFullYear(),
				now.getMonth(),
				now.getDate(),
				now.getHours(),
				now.getMinutes() + 1,
				0,
				0
			);
			const msUntilNext = nextMinute.getTime() - now.getTime();

			timers.current.timeout = setTimeout( () => {
				updateBg( true );
				timers.current.interval = setInterval(
					updateBg,
					TIMERS.UPDATE_INTERVAL
				);
			}, msUntilNext );

			// Clean up any previous interval
			if ( timers.current.interval ) {
				clearInterval( timers.current.interval );
			}
		}

		function clearTimers() {
			if ( timers.current.timeout ) {
				clearTimeout( timers.current.timeout );
			}
			if ( timers.current.interval ) {
				clearInterval( timers.current.interval );
			}
		}

		async function updateBg( force = false ) {
			if ( cancelled ) {
				return;
			}

			const now = new Date();
			setPreviewTime( now );

			const url = generateImageUrl(
				now,
				normalizedBasePath,
				normalizedExt,
				timezoneOffsetMinutes,
				useUpscaled
			);

			// Next minute
			const nextD = new Date( +now + TIMERS.UPDATE_INTERVAL );
			const nextUrl = generateImageUrl(
				nextD,
				normalizedBasePath,
				normalizedExt,
				timezoneOffsetMinutes,
				useUpscaled
			);

			if ( force || url !== lastGoodUrl ) {
				// Try to preload this bg
				try {
					await preload( url, TIMERS.PRELOAD_TIMEOUT );
					if ( ! cancelled ) {
						setBgUrl( url );
						setIsLoaded( true );
						setWarning( '' );
						lastGoodUrl = url;
					}
				} catch ( err ) {
					// background does not change; show warning
					setWarning(
						__(
							'Could not load background image: ',
							'teahouse-hero-block-wp'
						) + url
					);
					setIsLoaded( false );
					if ( url !== lastGoodUrl ) {
						// only log if different
						// eslint-disable-next-line no-console
						console.warn(
							`[teahouse-hero] Could not load: ${ url }`
						);
					}
				}
			}
			// Preload next (but don't set bgUrl)
			preload( nextUrl, TIMERS.PRELOAD_TIMEOUT ).catch( () => {} );
		}

		function onVisibilityChange() {
			if ( document.visibilityState === 'visible' ) {
				updateBg( true );
				// Clear and realign interval
				clearTimers();
				scheduleNextTick( new Date() );
			} else {
				clearTimers();
			}
		}

		clearTimers(); // whenever deps change

		updateBg( true ); // initial run

		scheduleNextTick( new Date() );

		document.addEventListener( 'visibilitychange', onVisibilityChange );

		return () => {
			cancelled = true;
			clearTimers();
			document.removeEventListener(
				'visibilitychange',
				onVisibilityChange
			);
		};
		// eslint-disable-next-line react-hooks/exhaustive-deps
	}, [
		normalizedBasePath,
		normalizedExt,
		timezoneOffsetMinutes,
		useUpscaled,
	] );

	// Apply noise styles whenever noise settings change
	useEffect( () => {
		if ( rootRef.current ) {
			applyNoiseStyles(
				rootRef.current,
				enableNoise,
				noiseOpacity,
				noiseScale,
				noiseType,
				noiseFrequency,
				noiseOctaves,
				noiseSeed,
				noiseColorType,
				noiseColor,
				noiseBlendMode
			);
		}
	}, [
		enableNoise,
		noiseOpacity,
		noiseScale,
		noiseType,
		noiseFrequency,
		noiseOctaves,
		noiseSeed,
		noiseColorType,
		noiseColor,
		noiseBlendMode,
	] );

	// Rely on core/Gutenberg alignment wrappers and our editor CSS.

	// Compose overlay
	const overlayStyle = {
		backgroundColor: overlayColor || DEFAULTS.OVERLAY_COLOR,
		opacity: typeof overlayOpacity === 'number' ? overlayOpacity : 0.4,
	};

	// Editor content alignment to CSS class
	let alignmentClass = 'is-content-center';
	if ( contentAlignment === 'top' ) {
		alignmentClass = 'is-content-top';
	} else if ( contentAlignment === 'bottom' ) {
		alignmentClass = 'is-content-bottom';
	}

	// Accessibility: region and optional aria-label
	const regionProps = { role: 'region' };
	if ( ariaLabel ) {
		regionProps[ 'aria-label' ] = ariaLabel;
	}

	// Build block props and merge classes instead of overwriting them
	const blockProps = useBlockProps( {
		ref: rootRef,
		style: {
			'--timeHeroBg': bgUrl ? `url(${ bgUrl })` : 'none',
		},
		'data-align': align || undefined,
		...regionProps,
	} );

	// Build alignment class
	let alignClass = '';
	if ( align === 'full' ) {
		alignClass = 'alignfull';
	} else if ( align === 'wide' ) {
		alignClass = 'alignwide';
	}

	return (
		<div
			{ ...blockProps }
			className={ `${
				blockProps.className
			} wp-block-emrikol-teahouse-hero ${ alignmentClass } ${
				isLoaded ? 'is-image-loaded' : ''
			} ${ alignClass }` }
		>
			<div className="teahouse-hero-overlay" style={ overlayStyle } />
			<div className="teahouse-hero-inner">
				<InnerBlocks
					template={ LOCALIZED_TEMPLATE }
					templateLock={ false }
				/>
			</div>
			<div className="teahouse-hero-badge">
				{ __(
					'Preview uses your (plus offset) local time.',
					'teahouse-hero-block-wp'
				) }
				<br />
				<small style={ { opacity: 0.7 } }>
					{ __( 'Current image:', 'teahouse-hero-block-wp' ) }{ ' ' }
					{ bgUrl
						? bgUrl.substring( bgUrl.lastIndexOf( '/' ) + 1 )
						: __( 'none', 'teahouse-hero-block-wp' ) }
				</small>
			</div>
			<InspectorControls>
				<PanelBody
					title={ __(
						'Background Image Settings',
						'teahouse-hero-block-wp'
					) }
				>
					<ToggleControl
						label={ __(
							'Use Upscaled Images',
							'teahouse-hero-block-wp'
						) }
						checked={ useUpscaled }
						onChange={ ( v ) =>
							setAttributes( { useUpscaled: v } )
						}
						help={ __(
							'Use higher resolution images from images-upscaled/ directory for better quality.',
							'teahouse-hero-block-wp'
						) }
					/>
					<TextControl
						type="number"
						label={ __(
							'Timezone Offset (minutes)',
							'teahouse-hero-block-wp'
						) }
						value={ timezoneOffsetMinutes ?? '' }
						onChange={ ( v ) =>
							setAttributes( {
								timezoneOffsetMinutes:
									v === '' ? null : parseInt( v, 10 ),
							} )
						}
						help={ __(
							'Optional offset (in minutes) added to local time before mapping.',
							'teahouse-hero-block-wp'
						) }
					/>
				</PanelBody>
				<PanelBody
					title={ __(
						'Content & Overlay',
						'teahouse-hero-block-wp'
					) }
					initialOpen={ false }
				>
					<ToolbarGroup>
						{ LOCALIZED_ALIGN_OPTIONS.map( ( opt ) => (
							<ToolbarButton
								key={ opt.value }
								isPressed={ contentAlignment === opt.value }
								onClick={ () =>
									setAttributes( {
										contentAlignment: opt.value,
									} )
								}
							>
								{ opt.label }
							</ToolbarButton>
						) ) }
					</ToolbarGroup>
					<label
						htmlFor="overlay-color-control"
						style={ { display: 'block', marginTop: '1em' } }
					>
						{ __( 'Overlay Color', 'teahouse-hero-block-wp' ) }
					</label>
					<ColorPalette
						id="overlay-color-control"
						value={ overlayColor }
						onChange={ ( color ) =>
							setAttributes( { overlayColor: color } )
						}
						disableCustomColors={ false }
					/>
					<RangeControl
						label={ __(
							'Overlay Opacity',
							'teahouse-hero-block-wp'
						) }
						value={ overlayOpacity }
						min={ 0 }
						max={ 1 }
						step={ 0.05 }
						onChange={ ( opacity ) =>
							setAttributes( { overlayOpacity: opacity } )
						}
					/>
					<ToggleControl
						label={ __(
							'Enable Noise Overlay',
							'teahouse-hero-block-wp'
						) }
						checked={ enableNoise }
						onChange={ ( v ) =>
							setAttributes( { enableNoise: v } )
						}
						help={ __(
							'Add a subtle noise texture to improve image appearance when enlarged.',
							'teahouse-hero-block-wp'
						) }
					/>
					{ enableNoise && (
						<>
							<RangeControl
								label={ __(
									'Noise Opacity (%)',
									'teahouse-hero-block-wp'
								) }
								value={ noiseOpacity }
								min={ 0 }
								max={ 100 }
								step={ 1 }
								onChange={ ( opacity ) =>
									setAttributes( { noiseOpacity: opacity } )
								}
								help={ __(
									'Set the noise overlay opacity as a percentage.',
									'teahouse-hero-block-wp'
								) }
							/>
							<SelectControl
								label={ __(
									'Noise Type',
									'teahouse-hero-block-wp'
								) }
								value={ noiseType }
								onChange={ ( type ) =>
									setAttributes( { noiseType: type } )
								}
								options={ [
									{
										label: __(
											'Fractal Noise',
											'teahouse-hero-block-wp'
										),
										value: 'fractalNoise',
									},
									{
										label: __(
											'Turbulence',
											'teahouse-hero-block-wp'
										),
										value: 'turbulence',
									},
								] }
								help={ __(
									'Choose the type of noise pattern to generate.',
									'teahouse-hero-block-wp'
								) }
							/>
							<RangeControl
								label={ __(
									'Noise Scale',
									'teahouse-hero-block-wp'
								) }
								value={ noiseScale }
								min={ 0 }
								max={ 100 }
								step={ 1 }
								onChange={ ( scale ) =>
									setAttributes( { noiseScale: scale } )
								}
								help={ __(
									'Control the size of noise particles (0-100%). Lower values create finer noise.',
									'teahouse-hero-block-wp'
								) }
							/>
							<RangeControl
								label={ __(
									'Noise Seed',
									'teahouse-hero-block-wp'
								) }
								value={ noiseSeed }
								min={ 1 }
								max={ 999 }
								step={ 1 }
								onChange={ ( seed ) =>
									setAttributes( { noiseSeed: seed } )
								}
								help={ __(
									'Random seed for noise pattern. Change for different patterns.',
									'teahouse-hero-block-wp'
								) }
							/>
							<SelectControl
								label={ __(
									'Noise Color Type',
									'teahouse-hero-block-wp'
								) }
								value={ noiseColorType }
								onChange={ ( colorType ) =>
									setAttributes( {
										noiseColorType: colorType,
									} )
								}
								options={ [
									{
										label: __(
											'Monochrome',
											'teahouse-hero-block-wp'
										),
										value: 'monochrome',
									},
									{
										label: __(
											'Full Color',
											'teahouse-hero-block-wp'
										),
										value: 'color',
									},
								] }
								help={ __(
									'Choose between monochrome or full-color noise.',
									'teahouse-hero-block-wp'
								) }
							/>
							{ noiseColorType === 'monochrome' && (
								<>
									<label
										htmlFor="noise-color-control"
										style={ {
											display: 'block',
											marginTop: '1em',
										} }
									>
										{ __(
											'Noise Color',
											'teahouse-hero-block-wp'
										) }
									</label>
									<ColorPalette
										id="noise-color-control"
										value={ noiseColor }
										onChange={ ( color ) =>
											setAttributes( {
												noiseColor: color || '#ffffff',
											} )
										}
										disableCustomColors={ false }
									/>
								</>
							) }
							<SelectControl
								label={ __(
									'Noise Blend Mode',
									'teahouse-hero-block-wp'
								) }
								value={ noiseBlendMode }
								onChange={ ( blendMode ) =>
									setAttributes( {
										noiseBlendMode: blendMode,
									} )
								}
								options={ [
									{
										label: __(
											'Normal',
											'teahouse-hero-block-wp'
										),
										value: 'normal',
									},
									{
										label: __(
											'Overlay',
											'teahouse-hero-block-wp'
										),
										value: 'overlay',
									},
									{
										label: __(
											'Multiply',
											'teahouse-hero-block-wp'
										),
										value: 'multiply',
									},
									{
										label: __(
											'Screen',
											'teahouse-hero-block-wp'
										),
										value: 'screen',
									},
									{
										label: __(
											'Darken',
											'teahouse-hero-block-wp'
										),
										value: 'darken',
									},
									{
										label: __(
											'Lighten',
											'teahouse-hero-block-wp'
										),
										value: 'lighten',
									},
									{
										label: __(
											'Color Dodge',
											'teahouse-hero-block-wp'
										),
										value: 'color-dodge',
									},
									{
										label: __(
											'Color Burn',
											'teahouse-hero-block-wp'
										),
										value: 'color-burn',
									},
									{
										label: __(
											'Hard Light',
											'teahouse-hero-block-wp'
										),
										value: 'hard-light',
									},
									{
										label: __(
											'Soft Light',
											'teahouse-hero-block-wp'
										),
										value: 'soft-light',
									},
									{
										label: __(
											'Difference',
											'teahouse-hero-block-wp'
										),
										value: 'difference',
									},
									{
										label: __(
											'Exclusion',
											'teahouse-hero-block-wp'
										),
										value: 'exclusion',
									},
									{
										label: __(
											'Hue',
											'teahouse-hero-block-wp'
										),
										value: 'hue',
									},
									{
										label: __(
											'Saturation',
											'teahouse-hero-block-wp'
										),
										value: 'saturation',
									},
									{
										label: __(
											'Color',
											'teahouse-hero-block-wp'
										),
										value: 'color',
									},
									{
										label: __(
											'Luminosity',
											'teahouse-hero-block-wp'
										),
										value: 'luminosity',
									},
								] }
								help={ __(
									'Choose how the noise overlay blends with the background image.',
									'teahouse-hero-block-wp'
								) }
							/>
							<RangeControl
								label={ __(
									'Noise Frequency',
									'teahouse-hero-block-wp'
								) }
								value={ noiseFrequency }
								min={ 0.1 }
								max={ 2.0 }
								step={ 0.1 }
								onChange={ ( frequency ) =>
									setAttributes( {
										noiseFrequency: frequency,
									} )
								}
								help={ __(
									'Base frequency for noise generation. Higher values create more detailed patterns.',
									'teahouse-hero-block-wp'
								) }
							/>
							<RangeControl
								label={ __(
									'Noise Octaves',
									'teahouse-hero-block-wp'
								) }
								value={ noiseOctaves }
								min={ 1 }
								max={ 8 }
								step={ 1 }
								onChange={ ( octaves ) =>
									setAttributes( { noiseOctaves: octaves } )
								}
								help={ __(
									'Number of noise layers. More octaves create more complex, detailed noise.',
									'teahouse-hero-block-wp'
								) }
							/>
						</>
					) }
					<TextControl
						label={ __(
							'Region Aria Label',
							'teahouse-hero-block-wp'
						) }
						value={ ariaLabel }
						onChange={ ( v ) => setAttributes( { ariaLabel: v } ) }
						help={ __(
							'For accessibility: e.g., "Hero banner". Leave blank for none.',
							'teahouse-hero-block-wp'
						) }
					/>
				</PanelBody>
			</InspectorControls>
			{ warning && (
				<Notice status="warning" isDismissible={ false }>
					{ warning }
				</Notice>
			) }
		</div>
	);
}
