/**
 * React hook that is used to mark the block wrapper element.
 * It provides all the necessary props like the class name.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-block-editor/#useblockprops
 */
import { useBlockProps, InnerBlocks } from '@wordpress/block-editor';

/**
 * The save function defines the way in which the different attributes should
 * be combined into the final markup, which is then serialized by the block
 * editor into `post_content`.
 *
 * @param {Object} root0            - The root object.
 * @param {Object} root0.attributes - Block attributes.
 * @see https://developer.wordpress.org/block-editor/reference-guides/block-api/block-edit-save/#save
 *
 * @return {Element} Element to render.
 */
export default function save( { attributes } ) {
	const {
		useUpscaled,
		timezoneOffsetMinutes,
		contentAlignment,
		overlayColor,
		overlayOpacity,
		ariaLabel,
		enableNoise,
		noiseOpacity,
		noiseScale,
		noiseType,
		noiseOctaves,
		noiseFrequency,
		noiseSeed,
		noiseColorType,
		noiseColor,
		noiseBlendMode,
	} = attributes;

	// Determine the final base path for upscaled images
	const finalBasePath = useUpscaled ? 'images-upscaled/' : 'images/';

	// CSS classes
	let alignmentClass = 'is-content-center';
	if ( contentAlignment === 'top' ) {
		alignmentClass = 'is-content-top';
	} else if ( contentAlignment === 'bottom' ) {
		alignmentClass = 'is-content-bottom';
	}

	// Overlay style
	const overlayStyle = {
		backgroundColor: overlayColor || '#000000',
		opacity: typeof overlayOpacity === 'number' ? overlayOpacity : 0.4,
	};

	// Region props for accessibility
	const regionProps = {
		role: 'region',
	};
	if ( ariaLabel ) {
		regionProps[ 'aria-label' ] = ariaLabel;
	}

	return (
		<div
			{ ...useBlockProps.save( {
				className: `wp-block-emrikol-teahouse-hero ${ alignmentClass }`,
				'data-base-path': finalBasePath,
				'data-file-ext': 'jpg',
				'data-tz-offset':
					typeof timezoneOffsetMinutes === 'number'
						? timezoneOffsetMinutes
						: '',
				'data-enable-noise': enableNoise,
				// Store noise opacity as percentage (0-100)
				'data-noise-opacity': noiseOpacity,
				'data-noise-scale': noiseScale,
				'data-noise-type': noiseType,
				'data-noise-frequency': noiseFrequency,
				'data-noise-octaves': noiseOctaves,
				'data-noise-seed': noiseSeed,
				'data-noise-color-type': noiseColorType,
				'data-noise-color': noiseColor,
				'data-noise-blend-mode': noiseBlendMode,
				style: {},
				...regionProps,
			} ) }
		>
			<div className="teahouse-hero-overlay" style={ overlayStyle } />
			<div className="teahouse-hero-inner">
				<InnerBlocks.Content />
			</div>
		</div>
	);
}
