/**
 * Image utilities for TeaHouse Hero block.
 *
 * Contains image preloading and path normalization logic.
 *
 * @package
 */

// Default timeout for image preloading (5 seconds)
export const DEFAULT_PRELOAD_TIMEOUT = 5000;

/**
 * Normalize base path for images.
 *
 * @param {string} path - Raw path input
 * @return {string} Normalized path with trailing slash
 */
export function normalizeBasePath( path ) {
	if ( ! path ) {
		return '';
	}
	let p = path.trim();

	// Handle images/ and images-upscaled/ paths
	if ( p.startsWith( 'images/' ) || p.startsWith( 'images-upscaled/' ) ) {
		const pluginUrl =
			( window.teaHouseHeroData && window.teaHouseHeroData.pluginUrl ) ||
			'';
		if ( pluginUrl ) {
			p = pluginUrl + p;
		}
	} else if ( ! /^https?:\/+/.test( p ) ) {
		p = 'https://' + p.replace( /^\/+/, '' );
	}

	if ( ! p.endsWith( '/' ) ) {
		p += '/';
	}
	return p;
}

/**
 * Normalize file extension.
 *
 * @param {string} ext - Raw extension input
 * @return {string} Normalized extension without leading dot
 */
export function normalizeFileExtension( ext ) {
	if ( ! ext ) {
		return '';
	}
	ext = ext.replace( /^\./, '' ).toLowerCase();
	return ext;
}

/**
 * Preload an image with timeout support.
 *
 * @param {string} src       - Image source URL
 * @param {number} timeoutMs - Timeout in milliseconds
 * @return {Promise} Promise that resolves when image loads or rejects on error/timeout
 */
export function preload( src, timeoutMs = DEFAULT_PRELOAD_TIMEOUT ) {
	return new Promise( ( resolve, reject ) => {
		const img = new window.Image();
		let done = false;
		const t = setTimeout( () => {
			if ( ! done ) {
				done = true;
				reject( new Error( 'timeout' ) );
			}
		}, timeoutMs );
		img.onload = () => {
			if ( ! done ) {
				done = true;
				clearTimeout( t );
				resolve();
			}
		};
		img.onerror = () => {
			if ( ! done ) {
				done = true;
				clearTimeout( t );
				reject( new Error( 'load-error' ) );
			}
		};
		img.src = src;
	} );
}
