# CLAUDE.md

This file provides guidance to Claude Code (claude.ai/code) when working with code in this repository.

## Development Commands

- `npm run build` - Build the block for production (creates optimized files in /build/)
- `npm run start` - Start development mode with hot reloading
- `npm run lint:js` - Lint JavaScript files
- `npm run lint:css` - Lint CSS/SCSS files
- `npm run packages-update` - Update WordPress packages
- `npm run plugin-zip` - Create plugin zip file

## Architecture Overview

This is a WordPress Gutenberg block plugin using the `@wordpress/create-block` framework. It displays a hero/cover section with background images that change based on the visitor's local time.

### Directory Structure

```
src/teahouse-hero/
├── block.json          - Block metadata and configuration
├── index.js           - Block registration entry point
├── edit.js            - Editor component with live preview and controls
├── render.php         - Server-side rendering for dynamic block
├── view.js            - Frontend JavaScript for time-based background switching
├── style.scss         - Frontend styles (compiled to style-index.css)
└── editor.scss        - Editor-specific styles (compiled to index.css)
```

### Core Components

- **teahouse-hero.php** - WordPress plugin entry point, registers blocks from build directory
- **src/teahouse-hero/block.json** - Block metadata and configuration (WordPress Block API v3)
- **src/teahouse-hero/index.js** - Block registration entry point  
- **src/teahouse-hero/edit.js** - Editor component with live preview and controls
- **src/teahouse-hero/render.php** - Server-side rendering for dynamic block output
- **src/teahouse-hero/view.js** - Frontend JavaScript for time-based background switching
- **src/teahouse-hero/style.scss** - Frontend styles
- **src/teahouse-hero/editor.scss** - Editor-specific styles

### Time Mapping Logic

The block uses a specific algorithm for mapping time to background images:
- At 03:14 exactly → `0314.jpg`
- From 03:16-03:59 → `04.jpg` 
- Even hours → `HH.jpg` (zero-padded)
- Odd hours → `(HH-1).jpg` (previous even hour)

Images are preloaded and update every minute with fade transitions. The logic is shared between editor preview and frontend view.

### Key Attributes

- `basePath` - Base URL for image files (default: images/ - resolves to plugin's images directory)
- `fileExtension` - Image format (default: jpg)
- `timezoneOffsetMinutes` - Optional timezone adjustment 
- `minHeight`, `contentAlignment`, `overlayColor`, `overlayOpacity` - Layout and styling
- `ariaLabel` - Accessibility label

### Images Directory

The plugin includes an `images/` directory for storing time-mapped background images:
- Use `images/` as basePath to reference the plugin's image directory
- Images are resolved server-side for security and performance
- See `images/README.md` for naming conventions and required files

### WordPress Integration

- Uses WordPress Scripts (@wordpress/scripts) for build tooling
- Follows WordPress coding standards and block development patterns
- Uses WordPress i18n for translations
- Integrates with WordPress block editor (Gutenberg)
- Supports InnerBlocks for rich content composition

### Architecture Principles
This plugin exemplifies **balanced KISS + SOLID design**:
- **3 focused classes** - Practical separation without over-engineering
- **Single responsibilities** - Each class has one reason to change
- **WordPress-native patterns** - No enterprise frameworks imposed
- **Composition over inheritance** - Simple dependency injection

## Project Philosophy

### Core Ethos: KISS (Keep It Simple, Stupid)

- **Simplicity over cleverness**: If there's a simple solution and a complex one, choose simple
- **Avoid premature optimization**: Build for clarity first, optimize when needed
- **Question complexity**: If a solution feels complicated, step back and find the simpler path
- **One tool, one purpose**: Prefer focused tools over swiss army knives
- **Configuration over code**: Simple config changes beat code modifications
- **Boring technology**: Choose proven, stable solutions over cutting-edge complexity

---

## Language & Dependencies

### Dependency Management
- **Minimize dependencies** - favor "vanilla" implementations
- **ALWAYS ASK** before installing new dependencies
- When dependencies are necessary, choose well-established, minimal ones
- Document dependency choices and rationale in code comments

---

## Code Quality & Architecture

### Readability & Maintenance
- **Write for novice developers** - assume code will be read by others
- **Favor readable code over clever shortcuts**
- **Comment extensively** about architectural and design decisions
- **Explain obtuse/obfuscated code** when unavoidable due to external constraints

### SOLID Principles Focus
- **Single Responsibility Principle**: A class should have one, and only one, reason to change
- **Split by responsibility, not size**: If a class handles business logic + I/O + formatting, it's too large regardless of line count
- **Design for change**: Consider what would cause a component to need modification

### Balancing KISS and SOLID
- **KISS wins over SOLID**: When in conflict, choose simplicity over architectural purity
- **Question every abstraction**: Interfaces, abstract classes, and complex patterns must justify their existence
- **Avoid over-engineering**: 2-3 focused classes are often better than 6+ "perfectly separated" classes
- **WordPress context matters**: Don't impose enterprise patterns on simple WordPress plugins
- **Practical separation**: Separate UI from logic, isolate complex database queries, keep business logic together
- **No premature abstraction**: Only create interfaces/abstractions when you have 2+ concrete implementations

### Modularity & Future-Proofing
- **Plugin architecture**: New platform parsers should integrate without core changes
- **Separation of concerns**: Keep parsing, indexing, and search as distinct modules
- **Configuration-driven**: Platform-specific settings in config files, not hardcoded

---

## Session & Context Management

### TODO.md Usage
- **Liberal updates** to track progress
- **Session continuity** - TODO.md is the source of truth if context is lost
- **Brain dump/scratchpad** for notes and intermediate thoughts
- **Mark completed tasks** with timestamps

### Context Optimization
- **Monitor context window usage**
- **Flag when sections are complete** and can be cleared
- **Optimize for performance** and reduce context poisoning
- **Clear completed work** to make room for new tasks

---

## Development Workflow

### Task Management
1. **Update TODO.md** before starting work
2. **Mark tasks in-progress** while working
3. **Complete tasks** with completion notes
4. **Flag context optimization opportunities**

### Code Review Principles
- **Self-document architectural decisions**
- **Comment non-obvious implementations**
- **Flag potential refactoring opportunities**
- **Consider modularity for future platforms**

### Decision Making & Recommendations
- **Be opinionated**: When presenting options, research and recommend the best choice
- **Provide evidence**: Back recommendations with research, benchmarks, or best practices
- **Default to expertise**: User relies on Claude's knowledge for technical decisions
- **Present reasoning**: Explain why a particular choice is recommended
- **Research before asking**: Use available tools to gather evidence for recommendations
- **Verify facts with tools**: Use bash commands, web searches, and file reads to verify information
- **Check dates and versions**: Always use `date` command and version checks rather than assumptions

---

*This document is living - update as development principles evolve.*

---

## Common Development Commands

### Setup and Dependencies
```bash
# Install dependencies
composer install

# All dependencies are included - no additional setup needed
```

### Code Quality

**IMPORTANT: Code Quality Process**

* Inline comments must end in full-stops, exclamation marks, or question marks
* Use Yoda Condition checks, you must
* Always add return type declrations for functions
* Always add type hinting for function arguments
* Always add `declare(strict_types=1);` strict type declaration to the top of PHP files
* Unless it needs to be private, all class objects should be public by default
* **Use PHP namespaces** - All PHP code should use a consistent namespace structure

### PHP Namespace Guidelines

**Project Namespace Structure:**
- **Primary Namespace**: `TeaHouseHero` for main plugin functionality
- **Render Namespace**: `TeaHouseHero\Render` for render.php files
- **Sub-namespaces**: Use logical groupings like `TeaHouseHero\Blocks`, `TeaHouseHero\Utils` if needed
- **Namespace Declaration**: Place immediately after `declare(strict_types=1);`

**Implementation Pattern:**
```php
<?php
/**
 * File description.
 *
 * @package TeaHouseHero
 */

declare( strict_types=1 );

namespace TeaHouseHero;

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// Functions and classes here
```

**Benefits:**
- **Conflict Prevention**: Eliminates global namespace pollution
- **Code Organization**: Clean, simple namespace structure
- **WordPress Standards**: Follows modern PHP and WordPress best practices
- **Future-Proofing**: Easier to refactor and extend

**Namespace Usage:**
- **Within namespace**: Call functions directly: `resolve_images_path()`
- **Outside namespace**: Use fully qualified: `TeaHouseHero\resolve_images_path()`
- **Use statements**: Import with `use TeaHouseHero\resolve_images_path;`

### Common PHPCS Errors and How to Prevent Them

**CRITICAL: Prevent These Errors by Following These Patterns**

These are the most common PHPCS violations encountered during development. Follow these patterns to write compliant code from the start:

#### 1. File Header Structure
```php
<?php
/**
 * File description here.
 *
 * More detailed description if needed.
 *
 * @package WordPress
 */

declare( strict_types=1 );

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}
```
**Key Points:**
- File docblock must immediately follow `<?php` tag
- `declare(strict_types=1);` comes AFTER the file docblock
- No blank line between `<?php` and docblock

#### 2. Translators Comments for Internationalization
```php
// translators: %s is the site name.
$message = sprintf( __( 'Welcome to %s!', 'pagegen' ), $site_name );

// translators: %d is the number of items, %s is the item type.
$text = sprintf( __( 'Found %d %s', 'pagegen' ), $count, $type );
```
**Key Points:**
- Add `// translators:` comment immediately before functions with placeholders
- Explain what each placeholder represents
- Required for `__()`, `_e()`, `esc_html__()`, `esc_attr__()` with `%s`, `%d`, etc.

#### 3. Output Escaping
```php
// Correct - all output is escaped
echo '<input type="text" value="' . esc_attr( $value ) . '">';
echo '<p>' . esc_html( $message ) . '</p>';
echo '<a href="' . esc_url( $link ) . '">' . esc_html( $text ) . '</a>';

// Incorrect - unescaped variables
echo '<input type="text" value="' . $value . '">'; // PHPCS error
echo '<option value="5" ' . $selected . '>'; // PHPCS error
```
**Key Points:**
- Use `esc_attr()` for HTML attributes
- Use `esc_html()` for HTML content
- Use `esc_url()` for URLs
- Use `checked()`, `selected()`, `disabled()` for form states

#### 4. Input Sanitization
```php
// Correct - sanitize all input
$hours = isset( $_POST['hours'] ) ? absint( $_POST['hours'] ) : 24;
$email = isset( $_POST['email'] ) ? sanitize_email( $_POST['email'] ) : '';
$text = isset( $_POST['message'] ) ? sanitize_text_field( $_POST['message'] ) : '';

// Incorrect - direct $_POST usage
$hours = $_POST['hours']; // PHPCS error
$limit = $_POST['limit']; // PHPCS error
```
**Key Points:**
- Always sanitize `$_POST`, `$_GET`, `$_REQUEST` data
- Use appropriate sanitization functions: `absint()`, `sanitize_text_field()`, `sanitize_email()`
- Check if input exists with `isset()` before sanitizing

#### 5. Inline Comments
```php
// Correct - ends with punctuation
$result = calculate_average( $data ); // Calculate the mean value.
$is_valid = check_status(); // Verify the current state!
$count = get_total(); // How many items do we have?

// Incorrect - no punctuation
$result = calculate_average( $data ); // Calculate the mean value
```

#### 6. Yoda Conditions
```php
// Correct - Yoda conditions
if ( 5 === $count ) { }
if ( 'active' === $status ) { }
if ( true === $is_enabled ) { }

// Incorrect - regular conditions  
if ( $count === 5 ) { } // PHPCS error
if ( $status === 'active' ) { } // PHPCS error
```

#### 7. Short Ternary Operators
```php
// Correct - full ternary
return $results ? $results : array();
$value = $input ? $input : 'default';

// Incorrect - short ternary
return $results ?: array(); // PHPCS error
```

#### 8. WordPress Options Autoload
```php
// Correct - explicitly disable autoload for plugin data
update_option( 'my_plugin_setting', $value, false );
add_option( 'my_plugin_data', $data, '', false );

// Incorrect - autoload enabled by default
update_option( 'my_plugin_setting', $value ); // Causes unnecessary autoloading
```

After making any code changes, always run these commands in order:

**For PHP files:**
1. **Auto-fix violations**: `composer fix` or `composer phpcbf path/to/file.php` (fixes what it can automatically)
2. **Check remaining issues**: `composer lint` or `composer phpcs path/to/file.php` (reports remaining violations)

**Note**: The `composer phpcs` and `composer phpcbf` commands automatically use the project's `vendor/bin/phpcs` if available in the git repository root.
3. **Fix manually**: Address any remaining PHPCS violations
4. **Never ignore**: Do not add `phpcs:ignore` or `phpcs:disable` comments unless the user explicitly requests it

**Example workflow:**
```bash
# Make code changes
# Then run:

# For PHP (direct commands):
composer fix                 # Auto-fix formatting, spacing, etc. in current directory
composer lint                # Check for remaining violations in current directory
# Or for specific files:
composer phpcbf path/to/file.php
composer phpcs path/to/file.php

# Fix any reported issues manually
# Ask user for guidance if unsure how to fix something
```

**Important Notes:**

**For PHP:**
- `phpcbf` (PHP Code Beautifier and Fixer) automatically fixes many formatting issues
- `phpcs` (PHP_CodeSniffer) reports remaining violations that need manual fixing
- Only add `phpcs:ignore` comments when the user specifically instructs you to do so
- Always ask the user for guidance if you're unsure how to fix a PHPCS violation

### Basic PHP Syntax Checking
```bash
# Check PHP syntax for all files
find . -name "*.php" -exec php -l {} \;

# Check and lint specific core files
php -l path/to/file.php
```

### Testing and Quality Assurance

**CRITICAL: Test Integrity Philosophy**

**DO THE WORK - DON'T CHEAT THE TESTS**

Tests are a critical part of our development cycle and must be treated with absolute integrity:

- **Fix the code, not the test** - When tests fail, the priority is to fix the underlying implementation
- **Tests reflect requirements** - Failing tests indicate the code doesn't meet specifications
- **No shortcuts or workarounds** - Never modify tests to pass without fixing the actual issue
- **Understand failures** - Investigate why tests fail before making any changes
- **Maintain test quality** - Tests should be as well-written and maintained as production code

**Test-Driven Development Approach:**
1. **Read the failing test** - Understand what behavior is expected
2. **Analyze the gap** - Identify what's missing or broken in the implementation
3. **Fix the implementation** - Make the minimal changes needed to satisfy the test
4. **Verify the fix** - Ensure tests pass for the right reasons
5. **Refactor if needed** - Improve code quality while keeping tests green

**When Tests Fail:**
- **Never ignore failing tests** - All tests must pass before considering work complete
- **Don't disable or skip tests** - Address the root cause instead
- **Don't modify test expectations** - unless requirements have genuinely changed
- **Document any test changes** - Explain why test modifications were necessary

```bash
# Run PHP unit tests using composer (recommended):
composer test

# or directly:
./run-tests.sh
```

### WordPress Options Guidelines

**IMPORTANT: WordPress Options Autoload Policy**

When using `update_option()` or `add_option()`, always explicitly set the `autoload` parameter to `false` unless the user specifically requests autoloading:

```php
// Correct - prevents unnecessary autoloading
update_option( 'my_plugin_setting', $value, false );
add_option( 'my_plugin_data', $data, '', false );

// Incorrect - causes autoloading on every page load
update_option( 'my_plugin_setting', $value );
add_option( 'my_plugin_data', $data );
```

**Rationale:**
- Plugin settings and data are rarely needed on every WordPress page load
- Autoloaded options increase memory usage and slow down site performance
- Most plugin data should only be loaded when specifically requested
- Only set autoload to `true` when the user explicitly requests it for frequently-accessed data

**Examples of options that should NOT autoload:**
- Configuration settings (API keys, remote site configs)
- Historical data (metrics, logs, cached results)
- Admin-only settings (debug flags, collection preferences)
- Large datasets (export data, bulk operations)

This applies to all options created or updated by the plugin.

### Code Quality Standards

**PHP Standards**
- Follows WordPress coding standards via `.phpcs.xml.dist`
- Minimum PHP 8.0 compatibility required
- Custom sniffs in `phpcs/` directory for additional validation
- Uupdate text domain in .phpcs.xml.dist if changed
- Prefixes in .phpcs.xml.dist for global functions/classes

---

## TODO.md Formatting Guidelines

### Proper TODO Format Structure

TODO.md should use checkable format with clear priority levels and organization:

**Priority Levels:**
- `🔴 HIGH PRIORITY (MUST FIX)` - Critical issues that must be addressed
- `🟡 MEDIUM PRIORITY (SHOULD FIX)` - Important improvements 
- `🟢 LOW PRIORITY (NICE TO HAVE)` - Optional enhancements

**Checkbox Format:**
```markdown
- [ ] **Task Name** - Description
  - Sub-task details
  - Specific requirements
- [x] **Completed Task** - What was accomplished
```

**Required Sections:**
1. **Priority-based task sections** with color coding
2. **VALIDATION TASKS** - Steps to run after each change
3. **DOCUMENTATION UPDATES** - What docs need updating
4. **ANALYSIS COMPLETED** - Checkoff completed research/analysis

**Example Structure:**
```markdown
# TODO: [Project Area]

## HIGH PRIORITY (MUST FIX) 🔴
- [ ] **Critical Task** - Why it's important
  - Specific implementation details
  - Methods/files to modify

## VALIDATION TASKS ✅
- [ ] **Run PHP syntax check**: `find . -name "*.php" -exec php -l {} \;`
- [ ] **Run PHPCS**: `phpcbf --extensions=php .`

## ANALYSIS COMPLETED ✅
- [x] **Research completed** - What was found
```

### TODO.md Best Practices

- **Use checkboxes** - `- [ ]` for pending, `- [x]` for completed
- **Be specific** - Include file paths, method names, line numbers
- **Add context** - Explain why tasks are needed
- **Update frequently** - Mark completed items immediately
- **Include validation** - Always have testing/quality check tasks
- **Reference locations** - Use `file:line` format for easy navigation