/**
 * Noise generation utilities for TeaHouse Hero block.
 *
 * Functions for generating dynamic SVG noise patterns.
 *
 * @package
 */

/**
 * Generate SVG noise pattern as data URI.
 *
 * @param {string} type       - Noise type ('fractalNoise' or 'turbulence').
 * @param {number} frequency  - Base frequency (0.1-2.0).
 * @param {number} octaves    - Number of octaves (1-8).
 * @param {number} seed       - Random seed (0-999).
 * @param {string} colorType  - Color type ('monochrome' or 'color').
 * @param {string} noiseColor - Noise color (hex color for monochrome).
 * @return {string} SVG data URI.
 */
export function generateNoiseSVG(
	type = 'fractalNoise',
	frequency = 0.9,
	octaves = 4,
	seed = 1,
	colorType = 'monochrome',
	noiseColor = '#ffffff'
) {
	// Generate unique filter ID using seed to avoid conflicts
	const filterId = `noiseFilter${ seed }`;

	let colorMatrix = '';
	if ( colorType === 'monochrome' ) {
		// Convert hex color to RGB values (0-1 range)
		const hex = noiseColor.replace( '#', '' );
		const r = parseInt( hex.substr( 0, 2 ), 16 ) / 255;
		const g = parseInt( hex.substr( 2, 2 ), 16 ) / 255;
		const b = parseInt( hex.substr( 4, 2 ), 16 ) / 255;

		// Create monochrome color matrix
		colorMatrix = `<feColorMatrix type="matrix" values="${ r } ${ g } ${ b } 0 0 ${ r } ${ g } ${ b } 0 0 ${ r } ${ g } ${ b } 0 0 0 0 0 1 0" />`;
	}

	const svg = `<svg viewBox="0 0 400 400" xmlns="http://www.w3.org/2000/svg">
		<filter id="${ filterId }">
			<feTurbulence 
				type="${ type }" 
				baseFrequency="${ frequency }" 
				numOctaves="${ octaves }" 
				seed="${ seed }"
				stitchTiles="stitch" 
			/>
			${ colorMatrix }
		</filter>
		<rect width="100%" height="100%" filter="url(#${ filterId })" />
	</svg>`;

	// Encode as data URI
	const encoded = encodeURIComponent( svg );
	return `url("data:image/svg+xml,${ encoded }")`;
}

/**
 * Apply noise styles to an element.
 *
 * @param {HTMLElement} element        - Target element.
 * @param {boolean}     enableNoise    - Whether to enable noise.
 * @param {number}      noiseOpacity   - Noise opacity (percentage 0-100).
 * @param {number}      noiseScale     - Noise scale as percentage (0-100).
 * @param {string}      noiseType      - Noise type.
 * @param {number}      noiseFrequency - Noise frequency.
 * @param {number}      noiseOctaves   - Noise octaves.
 * @param {number}      noiseSeed      - Random seed (0-999).
 * @param {string}      noiseColorType - Color type ('monochrome' or 'color').
 * @param {string}      noiseColor     - Noise color (hex).
 * @param {string}      noiseBlendMode - CSS blend mode for noise overlay.
 */
export function applyNoiseStyles(
	element,
	enableNoise,
	noiseOpacity,
	noiseScale,
	noiseType,
	noiseFrequency,
	noiseOctaves,
	noiseSeed = 1,
	noiseColorType = 'monochrome',
	noiseColor = '#ffffff',
	noiseBlendMode = 'overlay'
) {
	if ( ! enableNoise ) {
		element.style.setProperty( '--noiseOpacity', '0' );
		element.style.setProperty( '--noiseBg', 'none' );
		return;
	}

	// Convert percentage (0-100) to 0-1 for CSS opacity
	const normalizedOpacity = Math.max(
		0,
		Math.min( 1, ( noiseOpacity ?? 0 ) / 100 )
	);
	element.style.setProperty( '--noiseOpacity', String( normalizedOpacity ) );
	// Convert percentage to pixels: 0-100% maps to 25-200px range
	const scaleInPx = Math.round( 25 + ( noiseScale / 100 ) * 175 );
	element.style.setProperty( '--noiseScale', scaleInPx + 'px' );

	// Generate and apply dynamic noise background
	const noiseSVG = generateNoiseSVG(
		noiseType,
		noiseFrequency,
		noiseOctaves,
		noiseSeed,
		noiseColorType,
		noiseColor
	);
	element.style.setProperty( '--noiseBg', noiseSVG );
	element.style.setProperty( '--noiseBlendMode', noiseBlendMode );
}
