/**
 * Time mapping utilities for TeaHouse Hero block.
 *
 * Contains time calculation and filename generation logic
 * shared between editor and frontend components.
 *
 * @package
 */

/**
 * Compute time parts with optional timezone offset.
 *
 * @param {Date}   now             - Current date/time
 * @param {number} tzOffsetMinutes - Optional timezone offset in minutes
 * @return {Object} Object with hour and minute properties
 */
export function computeTimeParts( now, tzOffsetMinutes ) {
	// Copy the date to avoid mutation.
	let date = new Date( +now );
	if ( typeof tzOffsetMinutes === 'number' && ! isNaN( tzOffsetMinutes ) ) {
		date = new Date( date.getTime() + tzOffsetMinutes * 60000 );
	}
	const hour = date.getHours();
	const minute = date.getMinutes();
	return { hour, minute };
}

/**
 * Convert time to filename using TeaHouse Hero mapping algorithm.
 *
 * Algorithm:
 * - At 03:14 exactly → `0314.jpg`
 * - From 03:16-03:59 → `04.jpg`
 * - Even hours → `HH.jpg` (zero-padded)
 * - Odd hours → `(HH-1).jpg` (previous even hour)
 *
 * @param {number} hour          - Hour (0-23)
 * @param {number} minute        - Minute (0-59)
 * @param {string} fileExtension - File extension (e.g., 'jpg')
 * @return {string} Generated filename
 */
export function timeToFilename( hour, minute, fileExtension ) {
	// Special case: exactly 03:14
	if ( hour === 3 && minute === 14 ) {
		return `0314.${ fileExtension }`;
	}
	// 03:16 through 03:59 → 04
	if ( hour === 3 && minute >= 16 ) {
		return `04.${ fileExtension }`;
	}
	// Even hour: use as is
	if ( hour % 2 === 0 ) {
		return `${ String( hour ).padStart( 2, '0' ) }.${ fileExtension }`;
	}
	// Odd hour: previous even hour
	const prev = hour - 1;
	return `${ String( prev ).padStart( 2, '0' ) }.${ fileExtension }`;
}

/**
 * Generate full image URL from time components.
 *
 * @param {Date}    now             - Current date/time
 * @param {string}  basePath        - Base path for images
 * @param {string}  fileExtension   - File extension
 * @param {number}  tzOffsetMinutes - Optional timezone offset
 * @param {boolean} useUpscaled     - Whether to use upscaled images
 * @return {string} Complete image URL
 */
export function generateImageUrl(
	now,
	basePath,
	fileExtension,
	tzOffsetMinutes,
	useUpscaled = false
) {
	const { hour, minute } = computeTimeParts( now, tzOffsetMinutes );
	const filename = timeToFilename( hour, minute, fileExtension );

	// If using upscaled and basePath is 'images/', switch to 'images-upscaled/'
	let finalBasePath = basePath;
	if ( useUpscaled && basePath === 'images/' ) {
		finalBasePath = 'images-upscaled/';
	}

	return `${ finalBasePath }${ filename }`;
}
