/**
 * Finds and operates all Time-based Hero blocks on the front end.
 * Frontend logic for:
 *  - Computing filename per minute (with timezone offset)
 *  - Preloading images
 *  - Updating background-image via CSS variable (--timeHeroBg)
 *  - Class 'is-image-loaded' added/removed for fade effect.
 *  - Visibilitychange, interval handling, as per spec.
 */

import { generateImageUrl } from './utils/time-utils';
// Frontend-only styles are provided via block.json "viewStyle".
import {
	normalizeBasePath,
	normalizeFileExtension,
	preload,
} from './utils/image-utils';
import { applyNoiseStyles } from './utils/noise-utils';
import { TIMERS } from './constants';

// -------- Time mapping and utility logic moved to utils/ ---------

// Run after DOMContentLoaded
function handleTimeHeroBlock( el ) {
	const basePath = normalizeBasePath(
		el.getAttribute( 'data-base-path' ) || ''
	);
	const ext = normalizeFileExtension(
		el.getAttribute( 'data-file-ext' ) || 'jpg'
	);
	const tzOffset =
		el.hasAttribute( 'data-tz-offset' ) &&
		el.getAttribute( 'data-tz-offset' ) !== ''
			? parseInt( el.getAttribute( 'data-tz-offset' ), 10 )
			: null;

	// Read noise settings from data attributes
	const enableNoise = el.getAttribute( 'data-enable-noise' ) === 'true';
	// Treat opacity as percentage (0-100)
	const noiseOpacity = parseFloat(
		el.getAttribute( 'data-noise-opacity' ) || '100'
	);
	const noiseScale = parseInt(
		el.getAttribute( 'data-noise-scale' ) || '50',
		10
	);
	const noiseType = el.getAttribute( 'data-noise-type' ) || 'fractalNoise';
	const noiseFrequency = parseFloat(
		el.getAttribute( 'data-noise-frequency' ) || '0.9'
	);
	const noiseOctaves = parseInt(
		el.getAttribute( 'data-noise-octaves' ) || '4',
		10
	);
	const noiseSeed = parseInt(
		el.getAttribute( 'data-noise-seed' ) || '1',
		10
	);
	const noiseColorType =
		el.getAttribute( 'data-noise-color-type' ) || 'monochrome';
	const noiseColor = el.getAttribute( 'data-noise-color' ) || '#ffffff';
	const noiseBlendMode =
		el.getAttribute( 'data-noise-blend-mode' ) || 'overlay';

	let isMounted = true;
	const timers = { timeout: null, interval: null };
	let lastGoodUrl = '';

	function setBackground( url ) {
		el.style.setProperty( '--timeHeroBg', `url(${ url })` );
		el.classList.add( 'is-image-loaded' );
	}

	function clearBg() {
		el.style.removeProperty( '--timeHeroBg' );
		el.classList.remove( 'is-image-loaded' );
	}

	function scheduleNextTick( now ) {
		const nextMinute = new Date(
			now.getFullYear(),
			now.getMonth(),
			now.getDate(),
			now.getHours(),
			now.getMinutes() + 1,
			0,
			0
		);
		const msUntilNext = nextMinute.getTime() - now.getTime();

		timers.timeout = setTimeout( () => {
			updateBg( true );
			timers.interval = setInterval( updateBg, TIMERS.UPDATE_INTERVAL );
		}, msUntilNext );

		if ( timers.interval ) {
			clearInterval( timers.interval );
		}
	}

	function clearTimers() {
		if ( timers.timeout ) {
			clearTimeout( timers.timeout );
		}
		if ( timers.interval ) {
			clearInterval( timers.interval );
		}
	}

	async function updateBg( force = false ) {
		if ( ! isMounted ) {
			return;
		}

		const now = new Date();
		const url = generateImageUrl( now, basePath, ext, tzOffset );

		if ( force || url !== lastGoodUrl ) {
			try {
				await preload( url, TIMERS.PRELOAD_TIMEOUT );
				if ( ! isMounted ) {
					return;
				}
				setBackground( url );
				lastGoodUrl = url;
			} catch ( err ) {
				// Do not change bg and warn
				if ( url !== lastGoodUrl ) {
					// eslint-disable-next-line no-console
					console.warn( `[teahouse-hero] Could not load: ${ url }` );
				}
			}
		}

		// Preload next minute
		const nextD = new Date( +now + TIMERS.UPDATE_INTERVAL );
		const nextUrl = generateImageUrl( nextD, basePath, ext, tzOffset );
		preload( nextUrl, TIMERS.PRELOAD_TIMEOUT ).catch( () => {} );
	}

	function onVisibilityChange() {
		if ( document.visibilityState === 'visible' ) {
			updateBg( true );
			clearTimers();
			scheduleNextTick( new Date() );
		} else {
			clearTimers();
		}
	}

	function setup() {
		clearTimers();
		updateBg( true );
		scheduleNextTick( new Date() );
		document.addEventListener( 'visibilitychange', onVisibilityChange );

		// Apply noise styles
		applyNoiseStyles(
			el,
			enableNoise,
			noiseOpacity,
			noiseScale,
			noiseType,
			noiseFrequency,
			noiseOctaves,
			noiseSeed,
			noiseColorType,
			noiseColor,
			noiseBlendMode
		);
	}

	function destroy() {
		isMounted = false;
		clearTimers();
		document.removeEventListener( 'visibilitychange', onVisibilityChange );
		clearBg();
	}

	setup();

	// For unmount/SPA: expose destroy for GC if needed
	return destroy;
}

// Init
if ( typeof window !== 'undefined' && typeof document !== 'undefined' ) {
	window.addEventListener( 'DOMContentLoaded', () => {
		const heroes = document.querySelectorAll(
			'.wp-block-emrikol-teahouse-hero'
		);
		for ( const el of heroes ) {
			handleTimeHeroBlock( el );
		}
	} );
}
