<?php
/**
 * Plugin Name:       TeaHouse Hero Block
 * Description:       TeaHouse Hero block with a time‑based background image chosen according to the visitor's local time (client-side), updating every minute. Configurable path, timezone offset, file extension, overlay, and accessibility options.
 * Version:           1.0.0
 * Requires at least: 6.1
 * Requires PHP:      7.4
 * Author:            Derrick Tennant and Telex by WordPress.com
 * License:           GPL-2.0-or-later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:       teahouse-hero-block-wp
 *
 * @package TeaHouseHero
 */

declare( strict_types=1 );

namespace TeaHouseHero;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Resolves images/ path to the actual plugin URL.
 *
 * @param string $path The path that may contain images/.
 *
 * @return string The resolved URL
 */
function resolve_images_path( string $path ): string {
	if ( strpos( $path, 'images/' ) === 0 ) {
		$plugin_url = plugin_dir_url( __FILE__ );
		return str_replace( 'images/', $plugin_url . 'images/', $path );
	}
	if ( strpos( $path, 'images-upscaled/' ) === 0 ) {
		$plugin_url = plugin_dir_url( __FILE__ );
		return str_replace( 'images-upscaled/', $plugin_url . 'images-upscaled/', $path );
	}
	return $path;
}

/**
 * Registers the block using the metadata loaded from the `block.json` file.
 * Behind the scenes, it also registers all assets so they can be enqueued
 * through the block editor in the corresponding context.
 *
 * @see https://developer.wordpress.org/reference/functions/register_block_type/
 *
 * @return void
 */
function block_init(): void {
	register_block_type( __DIR__ . '/build/teahouse-hero' );
}
add_action( 'init', __NAMESPACE__ . '\block_init' );

/**
 * Add plugin URL to JavaScript for @images/ path resolution in editor.
 *
 * @return void
 */
function localize_editor_script(): void {
	wp_localize_script(
		'emrikol-teahouse-hero-editor-script',
		'teaHouseHeroData',
		array(
			'pluginUrl' => plugin_dir_url( __FILE__ ),
		)
	);
}
add_action( 'enqueue_block_editor_assets', __NAMESPACE__ . '\localize_editor_script' );

/**
 * Add plugin URL to JavaScript for @images/ path resolution in frontend.
 *
 * @return void
 */
function localize_frontend_script(): void {
	wp_localize_script(
		'emrikol-teahouse-hero-view-script',
		'teaHouseHeroData',
		array(
			'pluginUrl' => plugin_dir_url( __FILE__ ),
		)
	);
}
add_action( 'wp_enqueue_scripts', __NAMESPACE__ . '\localize_frontend_script' );
